<?php
/*
Copyright (c)2002-2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK.

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

LMS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with LMS; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/


/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright		Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK File Change Class
 *
 * This classe is used to manage things like backup of 
 * the configuration file if modified, writing to and so on...
 *
 * @package		dftk
 * @subpackage	dftk-file
 * @author		Rtp <rtp@rtp-net.org>
 *
 * @access		public
 */
class DftkFileChange
{
	/**
	 * Trace Manager
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_tracemgr;

	/**
	 * Configuration file name
	 *
	 * @access		private
	 * @var			string
	 */
	var $_Filename;

	/**
	 * File path
	 *
	 * @access		private
	 * @var			string
	 */
	var $_Path;

	/**
	 * File pointer
	 *
	 * @access		private
	 * @var			resource
	 */
	var $_Fp;

	/**
	 * Backup file extenstion
	 *
	 * @access		private
	 * @var			string
	 */
	var $_BackExt;

	/**
	 * Constructor
	 *
	 * @access public
	 *
	 * @param		object DftkDaTraceManager	&$tracemgr			Language Manager
	 * @param		string				$file				Configuration file name
	 * @param		string				$path				File path
	 * @param		string				$ext				Backup extenstion
	 */
	function DftkFileChange(&$tracemgr, $file, $path="./", $ext="bak")
	{
		$this->_tracemgr =& $tracemgr;

		if (!$this->_tracemgr->is_module("DFTK-FILE"))
			$this->_tracemgr->register_traces("DFTK-FILE",DFTK_ROOT_PATH."/file/traces/");

		$this->_Filename=$file;

		$this->_Path=$path;
		/* Be sure that the path is ended with a slash */
		if (strcmp(substr($this->_Path,-1),'/') != 0) 
			$this->_Path=$this->_Path."/";
		
		$this->_BackExt=".".$ext;

		$this->_Fp = false;
	}

	/**
	 * Open the file to handle
	 *
	 * @access		public
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &open()
	{
		$r =& $this->_tracemgr->create_trace();
		
		if ($this->_Fp)
			$r->add_event('dftk-file_alrop');
		else
		{
			$origfile = $this->_Path.$this->_Filename;
			$myfile = $origfile.".tmp";

			$this->_Fp=fopen($myfile,"w");
			if (!$this->_Fp)
				$r->add_event('dftk-file_ioerr');
		}

		return $r;
	}

	/**
	 * Close the file
	 *
	 * Result :
	 *  + 'file_changed' : true if the file has changed
	 *
	 * @access		public
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &close()
	{
		$r =& $this->_tracemgr->create_trace();

		if ($this->_Fp)
		{
			if (!fclose($this->_Fp))
				$r->add_event('dftk-file_ioerr');
			else
			{
				$origfile = $this->_Path.$this->_Filename;
				$backupfile = $origfile.$this->_BackExt;
				$myfile = $origfile.".tmp";

				$maj = $this->_is_maj();
				$r->set_result('file_changed', $maj);

				if ($maj) 
				{
					if (file_exists($origfile))
						if (!copy($origfile, $backupfile))
							$r->add_event('dftk-file_ioerr');
					if (!$r->has_error())
						if (!copy($myfile, $origfile))
							$r->add_event('dftk-file_ioerr');
				}

				if (!unlink($myfile))
					$r->add_event('dftk-file_ioerr');
			}
		}
		else
			$r->add_event('dftk-file_notop');

		return $r;
	}

	/**
	 * Write to the file
	 *
	 * Result :
	 *  + 'writen_bytes' : number of bytes really written
	 *
	 * @access		public
	 * @param		string					$string					String to write 
	 * @param		integer					$length					Number of bytes to write
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &write($string, $length=0)
	{
		if ($this->_Fp)
		{
			$r =& $this->_tracemgr->create_trace();
		
			if ($length)
				$n = fwrite($this->_Fp,$string,$length);
			else
				$n = fwrite($this->_Fp,$string);

			if ($n)
				$r->set_result('writen_bytes', $n);
			else
				$r->add_event('dftk-file_ioerr');
		}
		else
			$r->add_event('dftk-file_notop');

		return $r;
	}

	/**
	 * Check if the file was updated
	 *
	 * @access		private
	 * @return		boolean					$err					True if the file was updated
	 */
	function _is_maj() 
	{
		$origfile = $this->_Path.$this->_Filename;
		$myfile = $origfile.".tmp";


		if (!file_exists($origfile))
			return true;

		/* php versions prior to 4.2.0 don't have a md5_file function */
		if (!function_exists('md5_file'))
		{
			$orig_md5=md5(implode("", file($origfile)));
			$my_md5  =md5(implode("", file($myfile)));
		}
		else
		{
			$orig_md5=md5_file($origfile);
			$my_md5  =md5_file($myfile);
		}

		if ($orig_md5 != $my_md5)
			return true;
		
		return false;
	}
}
?>
